using System.Collections.Generic;
using UnityEngine;
using UnityEngine.SceneManagement;
using UnityEngine.XR.ARFoundation;
using UnityEngine.XR.ARSubsystems;
using TMPro;

[RequireComponent(typeof(ARRaycastManager))]
public class CylinderMeasureTool : MonoBehaviour
{
    [Header("Prefabs")]
    public GameObject reticlePrefab;             // reticle
    public GameObject cylinderIndicatorPrefab;   // prefab cylinder transparan
    public TextMeshPro resultTextPrefab;         // teks hasil hitungan

    private ARRaycastManager raycastManager;
    private GameObject reticle;

    private GameObject cylinder;   // instance cylinder
    private TextMeshPro resultText;

    private Vector3 point1;        // sisi pertama lingkaran alas
    private Vector3 point2;        // sisi kedua lingkaran alas
    private Vector3 centerPoint;   // titik pusat hasil midpoint
    private Vector3 topPoint;      // titik atas (tinggi)

    private int tapCount = 0;

    private static List<ARRaycastHit> hits = new List<ARRaycastHit>();

    public void LoadHome()
    {
        SceneManager.LoadScene("Home");
        // Debug.Log("Home");
    }

    void Start()
    {
        raycastManager = GetComponent<ARRaycastManager>();

        // spawn reticle
        reticle = Instantiate(reticlePrefab);
        reticle.SetActive(false);

        // spawn result text
        if (resultTextPrefab != null)
        {
            resultText = Instantiate(resultTextPrefab, Vector3.zero, Quaternion.identity);
            resultText.gameObject.SetActive(false);
        }
    }

    void Update()
    {
        UpdateReticle();

        if (Input.touchCount > 0 && Input.GetTouch(0).phase == TouchPhase.Began)
        {
            if (reticle.activeSelf) HandleTap();
        }

        // preview
        if (tapCount == 1 && reticle.activeSelf)
        {
            // preview diameter (antara point1 dan reticle)
            Vector3 tempCenter = (point1 + reticle.transform.position) / 2f;
            float radius = Vector3.Distance(point1, reticle.transform.position) / 2f;
            UpdateCylinder(tempCenter, radius, 0.001f);
        }
        else if (tapCount == 2 && reticle.activeSelf && cylinder != null)
        {
            float radius = Vector3.Distance(point1, point2) / 2f;

            float baseY = point1.y;
            float height = Mathf.Abs(reticle.transform.position.y - baseY);

            UpdateCylinder(centerPoint, radius, height);
        }


    }

    void UpdateReticle()
    {
        Vector2 screenCenter = new Vector2(Screen.width / 2, Screen.height / 2);
        Ray ray = Camera.main.ScreenPointToRay(screenCenter);

        if (tapCount < 2)
        {
            // Cari alas di plane
            if (raycastManager.Raycast(screenCenter, hits, TrackableType.PlaneWithinPolygon))
            {
                Pose hitPose = hits[0].pose;
                reticle.SetActive(true);
                reticle.transform.position = hitPose.position;
                reticle.transform.rotation = hitPose.rotation;
            }
            else
            {
                reticle.SetActive(false);
            }
        }
        else
        {
            // Tahap tinggi → reticle "melayang" mengikuti arah kamera
            reticle.SetActive(true);
            reticle.transform.position = ray.origin + ray.direction * 1.0f; // 1 meter di depan kamera
        }
    }


    void HandleTap()
    {
        tapCount++;

        if (tapCount == 1)
        {
            // sisi pertama
            point1 = reticle.transform.position;
        }
        else if (tapCount == 2)
        {
            // sisi kedua → hitung center & radius
            point2 = reticle.transform.position;
            centerPoint = (point1 + point2) / 2f;
            float radius = Vector3.Distance(point1, point2) / 2f;

            // spawn cylinder
            if (cylinder != null) Destroy(cylinder);
            cylinder = Instantiate(cylinderIndicatorPrefab, centerPoint, Quaternion.identity);

            UpdateCylinder(centerPoint, radius, 0.05f); // tinggi kecil dulu
        }
        else if (tapCount == 3)
        {
            topPoint = reticle.transform.position;
            float radius = Vector3.Distance(point1, point2) / 2f;

            float baseY = point1.y;
            float height = Mathf.Abs(topPoint.y - baseY);

            UpdateCylinder(centerPoint, radius, height);
            ShowResult(radius, height);

            tapCount = 0; // reset
        }

    }


    void UpdateCylinder(Vector3 center, float radius, float height)
    {
        if (cylinder == null) return;

        float baseY = point1.y; // alas tetap
        Vector3 pos = new Vector3(center.x, baseY + height / 2f, center.z);
        cylinder.transform.position = pos;

        cylinder.transform.localScale = new Vector3(radius * 2f, height / 2f, radius * 2f);
    }




    void ShowResult(float radius, float height)
    {
        float volume = Mathf.PI * radius * radius * height;
        float surface = 2 * Mathf.PI * radius * (radius + height);

        // konversi ke cm
        float rCm = radius * 100f;
        float hCm = height * 100f;
        float volCm = volume * 1000000f;
        float surfCm = surface * 10000f;

        string result = $"R = {rCm:F1} cm\nH = {hCm:F1} cm\nV = {volCm:F1} cm³\nA = {surfCm:F1} cm²";

        resultText.text = result;
        resultText.transform.position = cylinder.transform.position + Vector3.up * (height / 2f + 0.05f);
        resultText.gameObject.SetActive(true);

        Camera cam = Camera.main;
        if (cam != null)
        {
            resultText.transform.rotation =
                Quaternion.LookRotation(resultText.transform.position - cam.transform.position);
        }
    }
}
